package breadboards;

import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;

/**
 * Subclass of GObject used for displaying ovals
 * @author paul oser
 */
public class GLabel extends GObject {
  
  String text;
  Color color;
  String font = "TimesRoman";
  int fontSize = 12;

  /**
   * constructs a GLabel with specified text with lower left corner at (0,0)
   * @param text the specified text
   */
  public GLabel(String text) {
    this.text = text;
    setDefaults();
  }
  
  /**
   * constructs a GLabel with a specified text at a specified location
   * @param x the x-coordinate of the lower left corner of the GLabel
   * @param y the y-coordinate of the lower left corner of the GLabel
   * @param text the specified text
   */
  public GLabel(double x, double y, String text) {
    this.x = x;
    this.y = y;
    this.text = text;
    setDefaults();
  }
  
  /////////////
  // Getters //
  /////////////
  
  /**
   * returns the text of this GLabel as a string
   * @return the text of this GLabel
   */
  public String getText() {
    return this.text;
  }
  
  /**
   * returns an empty bounding rectangle for this GLabel
   * @return an empty bounding rectangle for this GLabel
   */
  public GRectangle getBounds() {
    return new GRectangle(this.x, this.y, 0,0);
  }
  //TODO: Should this be empty?  Probably a better way
  
  /**
   * returns the border color of this GLabel
   * @return the border color of this GLabel
   */
  public Color getColor() {
    return this.color;
  }
  
/* EXAMPLE TO REFERENCE:
//get metrics from the graphics
FontMetrics metrics = graphics.getFontMetrics(font);
//get the height of a line of text in this
//font and render context
int hgt = metrics.getHeight();
//get the advance of my text in this font
//and render context
int adv = metrics.stringWidth(text);
//calculate the size of a box to hold the
//text with some padding.
Dimension size = new Dimension(adv+2, hgt+2);
*/
  
  /**
   * returns the height of this GLabel
   * @return the height of this GLabel
   */
  public double getHeight() {
    Graphics g = this.getBreadboard().getGraphics();
    FontMetrics metrics = g.getFontMetrics(new Font(font, Font.PLAIN, fontSize));
    return metrics.getHeight()+2;
  }
  
  /**
   * returns the width of this GLabel
   * @return the width of this GLabel
   */
  public double getWidth() {
    Graphics g = this.getBreadboard().getGraphics();
    FontMetrics metrics = g.getFontMetrics(new Font(font, Font.PLAIN, fontSize));
    return metrics.stringWidth(text)+2;
  }
  
  /////////////
  // Setters //
  /////////////
  
  /**
   * sets the text of this GLabel
   * @param text the new text for this GLabel
   */
  public void setText(String text) {
    this.text = text;
  }
  
  /**
   * sets the font size of the GLabel
   * @param fontSize the specified font size
   */
  public void setFontSize(int fontSize) {
    this.fontSize = fontSize;
  }
  
  /** sets the font of the GLabel
   * @param font a string representing the font (like "TimesRoman")
   */
  public void setFont(String font) {
    this.font = font;
  }
  
  /**
   * sets the border color of the GLabel to the specified color
   * @param color the specified color
   */
  public void setColor(Color color) {
    this.color = color;
  }
 
  /** 
   * draws the GLabel (generally not called directly)
   */
  public void draw(Graphics g) {
    Color originalFillColor = g.getColor();
    
    g.setFont(new Font(font, Font.PLAIN, fontSize));
    g.setColor(color);    
    g.drawString(text, (int) (this.getX()), (int) (this.getY()));
    
    g.setColor(originalFillColor);
  }
  
  /**
   * returns true if (x,y) is inside this GLabel
   * @param x the x-coordinate of the point in question
   * @param y the y-coordinate of the point in question
   * @return true if (x,y) is inside this GLabel, false otherwise
   */
  @Override
  public boolean contains(double x, double y) {
    return ((x >= this.getX()) && (x <= this.getX() + this.getWidth()) &&
            (y >= this.getY()) && (y <= this.getY() + this.getHeight()));
  }
  
  /////////////////////
  // private methods //
  /////////////////////
  
  private void setDefaults() {
    this.color = Color.black;
  }
}
