package breadboards;

import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.util.ArrayList;

/**
 * Abstract class suitable for graphics objects that can be added to a Breadboard object
 * @author paul oser
 *
 */
public abstract class GObject implements Drawable {
  
  ////////////////////////
  // instance variables //
  ////////////////////////
  
  GObjectParent parent;
  
  double x;
  double y;
  
  boolean isVisible;
  
  ArrayList<MouseListener> mouseListeners;
  ArrayList<MouseMotionListener> mouseMotionListeners;
  
  //////////////////
  // constructors //
  //////////////////
  
  /**  
   * (only called through constructor chaining) 
   */
  public GObject() {
    this.isVisible = true;
    mouseListeners = new ArrayList<MouseListener>();
    mouseMotionListeners = new ArrayList<MouseMotionListener>();
  }
  
  //////////////////////
  // abstract methods //
  //////////////////////
  
  /**
   * required of subclasses to determine if the point (x,y) is contained in/on the GObject in question
   * @param x x-coordinate of the point in question
   * @param y y-coordinate of the point in question
   * @return true if the point is contained in the GObject, false otherwise
   */
  public abstract boolean contains(double x, double y);
  
  ///////////////////////////////////////
  // methods that should be overridden //
  ///////////////////////////////////////
  
  /**
   * required of subclasses to determine the bounding rectangle of the GObject in question
   * @return the bounding rectangle of the GObject
   */
  public abstract GRectangle getBounds();

  /////////////
  // getters //
  /////////////
  
  /**
   * returns the x-coordinate of the GObject
   * @return the x-coordinate of the GObject
   */
  public double getX() {
    return x;
  }
  
  /**
   * returns the y-coordinate of the GObject
   * @return the y-coordinate of the GObject
   */
  public double getY() {
    return y;
  }
  
  /**
   * returns the location of the GObject as a GPoint
   * @return the location of the GObject
   */
  public GPoint getLocation() {
    return new GPoint(this.getX(),this.getY());
  }
  
  /**
   * returns the parent of the GObject - typically a GCompound or a Breadboard object
   * @return the parent of the GObject
   */
  public GObjectParent getParent() {
    return this.parent;
  }
  
  /** 
   * returns the breadboard associated with this GObject -- or null if there is no such breadboard
   * @return the breadboard associated with this GObject -- or null if there is no such breadboard 
   */
  public Breadboard getBreadboard() {
    GObjectParent p = this.getParent();
    while (p instanceof GCompound) {
       p = ((GCompound) p).getParent();
    }
    return ((p instanceof Breadboard) ? (Breadboard) p : null);
  }
  
  /**
   * returns whether or not this GObject is visible
   * @return true if the GObject is visible, false otherwise
   */
  public boolean isVisible() {
    return this.isVisible;
  }
  
  /** 
   * sets the parent for this object to be the specified parent - note: it does not tell the parent of this child
   * @param parent the specified parent
   */
  public void setParent(GObjectParent parent) {
    this.parent = parent;
  }
  
  /////////////
  // setters //
  /////////////
  
  /**
   * sets the x-coordinate for this GObject to the specified x-coordinate
   * @param x the specified x-coordinate
   */
  public void setX(double x) {
    this.x = x;
    updateBreadboard();
  }
  
  /**
   * sets the y-coordinate for this GObject to the specified y-coordinate
   * @param y the specified y-coordinate
   */
  public void setY(double y) {
    this.y = y;
    updateBreadboard();
  }
  
  /**
   * sets the location of this GObject to (x,y)
   * @param x the new x-coordinate for this GObject
   * @param y the new y-coordinate for this GObject
   */
  public void setLocation(double x, double y) {
    this.setX(x);
    this.setY(y);
    updateBreadboard();
  }
  
  /**
   * sets whether this GObject is visible or not
   * @param isVisible true if the object is visible, false otherwise
   */
  public void setVisible(boolean isVisible) {
    this.isVisible = isVisible;
    updateBreadboard();
  }
  
  /**
   * Swaps the z-orders of this GObject and the object immediately "above" it on this GObject's parent object
   */
  public void sendForward() {
    this.getParent().sendForward(this);
  }
  
  /**
   * Swaps the z-orders of this GObject and the object immediately "below" it on this GObject's parent object
   */
  public void sendBackward() {
    this.parent.sendBackward(this);
  }
  
  /**
   * Moves this GObject "on top" of all other GObjects on this GObject's parent object
   */
  public void sendToFront() {
    this.parent.sendToFront(this);
  }
  
  /**
   * Moves this GObject "below" all other GObjects on this GObject's parent object
   */
  public void sendToBack() {
    this.parent.sendToBack(this);
  }
  
  ///////////////////
  // other methods //
  ///////////////////
  
  /**
   * adds a specified MouseListener to this GObject
   * @param mouseListener the MouseListener specified
   */
  public void addMouseListener(MouseListener mouseListener) {
    mouseListeners.add(mouseListener);
  }
  
  /**
   * adds a specified mouseMotionListener to this GObject
   * @param mouseMotionListener the MouseMotionListener specified
   */
  public void addMouseMotionListener(MouseMotionListener mouseMotionListener) {
    mouseMotionListeners.add(mouseMotionListener);
  }
  
  /**
   * removes a specified MouseListener from this GObject
   * @param mouseListener the specified MouseListener
   */
  public void removeMouseListener(MouseListener mouseListener) {
    mouseListeners.remove(mouseListener);
  }
  
  /**
   * removes a specified MouseMotionListener from this GObject
   * @param mouseMotionListener the specified MouseMotionListener
   */
  public void removeMouseMotionListener(MouseMotionListener mouseMotionListener) {
    mouseMotionListeners.remove(mouseMotionListener);
  }
  
  /**
   * gets an iterable collection of all MouseListeners associated with this GObject
   * @return an iterable collection of all MouseListeners associated with this GObject
   */
  public Iterable<MouseListener> getMouseListeners() {
    return mouseListeners;
  }
  
  /**
   * gets an iterable collection of all MouseMotionListeners associated with this GObject
   * @return an iterable collection of all MouseMotionListeners associated with this GObject
   */
  public Iterable<MouseMotionListener> getMouseMotionListeners() {
    return mouseMotionListeners;
  }
  
  /**
   * moves this GObject dx units horizontally and dy units vertically
   * @param dx how much to move this GObject horizontally (positive is to the right, negative to the left)
   * @param dy how much to move this GObject vertically (positive is down, negative is up)
   */
  public void move(double dx, double dy) {
    x += dx;
    y += dy;
    updateBreadboard();
  }
  
  /**
   * moves this GObject in a polar manner, r units from its current position in the theta direction
   * @param r how far to move from the current position
   * @param theta the direction in which this GObject is moved (expressed in radians)
   */
  public void movePolar(double r, double theta) {
    x += r * Math.cos(theta);
    y += r * Math.sin(theta);
  }
  
  /**
   * updates display of containing breadboard (should be called when the GObject changes state)
   */
  public void updateBreadboard() {
    if (this.parent != null)
      parent.updateDisplay();
  }
 
}
