package breadboards;

import java.awt.Color;
import java.awt.Graphics;

/**
 * Subclass of GObject used for displaying ovals
 * @author paul oser
 */
public class GOval extends GObject {
  
  double width;
  double height;
  
  boolean isFilled;
  Color color;
  Color fillColor;

  /**
   * constructs a GOval with specified width and height with upper left corner at (0,0)
   * @param width the specified width
   * @param height the specified height
   */
  public GOval(double width, double height) {
    this.width = width;
    this.height = height;
    setDefaults();
  }
  
  /**
   * constructs a GOval with a specified width and height at a specified location
   * @param x the x-coordinate of the upper left corner of the GOval
   * @param y the y-coordinate of the upper left corner of the GOval
   * @param width the specified width
   * @param height the specified height
   */
  public GOval(double x, double y, double width, double height) {
    this.x = x;
    this.y = y;
    this.width = width;
    this.height = height;
    setDefaults();
  }
  
  /////////////
  // Getters //
  /////////////
  
  /**
   * returns the bounding rectangle for this GOval
   * @return the bounding rectangle for this GOval
   */
  public GRectangle getBounds() {
    return new GRectangle(this.x, this.y, this.width, this.height);
  }
  
  /**
   * returns the border color of this GOval
   * @return the border color of this GOval
   */
  public Color getColor() {
    return this.color;
  }
  
  /**
   * returns the fill color for this GOval
   * @return the fill color for this GOval
   */
  public Color getFillColor() {
    return this.color;
  }
  
  /**
   * returns the height of this GOval
   * @return the height of this GOval
   */
  public double getHeight() {
    return this.height;
  }
  
  /**
   * returns the width of this GOval
   * @return the width of this GOval
   */
  public double getWidth() {
    return this.width;
  }
  
  /**
   * returns whether or not this GOval should be displayed with an opaque interior 
   * @return true if an opaque interior should be used, false otherwise
   */
  public boolean isFilled() {
    return this.isFilled;
  }
  
  /////////////
  // Setters //
  /////////////
  
  /**
   * sets the border color of the GOval to the specified color
   * @param color the specified color
   */
  public void setColor(Color color) {
    this.color = color;
  }
  
  /**
   * this must be called for the GOval to show an opaque interior
   * @param isFilled true if an opaque interior is to be displayed, false otherwise
   */
  public void setFilled(boolean isFilled) {
    this.isFilled = isFilled;
  }
  
  /**
   * set the fill color for the opaque interior to the specified color
   * @param fillColor the specified color
   */
  public void setFillColor(Color fillColor) {
    this.fillColor = fillColor;
  }
  
  /** 
   * set the width and height of the GOval to the specified values
   * @param width the specified width
   * @param height the specified height
   */
  public void setSize(double width, double height) {
    this.width = width;
    this.height = height;
  }
  
  /**
   * set the width, height, and location of the GOval to the specified values
   * @param x x-coordinate of the upper left corner of the GOval
   * @param y y-coordinate of the upper left corner of the GOval
   * @param width width the specified width
   * @param height height the specified height
   */
  public void setBounds(double x, double y, double width, double height) {
    this.x = x;
    this.y = y;
    this.width = width;
    this.height = height;
  }
  
  /**
   * set the width, height, and location of the GOval to those associated with the specified bounding rectangle 
   * @param bounds a bounding rectangle for the GOval with the same dimensions and location
   */
  public void setBounds(GRectangle bounds) {
    this.setBounds(bounds.getX(),bounds.getY(),bounds.getWidth(),bounds.getHeight());
  }
  
  /**
   * scale the width of the GOval by a factor of sx and the height of the GOval by a factor of sy
   * @param sx the horizontal scaling factor
   * @param sy the vertical scaling factor
   */
  public void scale(double sx, double sy) {
    this.width *= sx;
    this.height *= sy;
  }
  
  /** 
   * scale this GOval by a factor of sf
   * @param sf the scaling factor
   */
  public void scale(double sf) {
    this.scale(sf,sf);
  }
  
  /** 
   * draws the GOval (generally not called directly)
   */
  public void draw(Graphics g) {
    Color originalFillColor = g.getColor();
    
    if (this.isFilled) { 
      g.setColor(fillColor);
      g.fillOval((int) x, (int) y, (int) width, (int) height); 
    }
    
    g.setColor(color);
    g.drawOval((int) x, (int) y, (int) width, (int) height); 
    
    g.setColor(originalFillColor);
  }
  
  /**
   * returns true if (x,y) is inside this GRect
   * @param x the x-coordinate of the point in question
   * @param y the y-coordinate of the point in question
   * @return true if (x,y) is inside this GRect, false otherwise
   */
  @Override
  public boolean contains(double x, double y) {
    double cx = this.x + this.width/2.0;
    double cy = this.y + this.height/2.0;
    double x0 = 2*(x-cx) / width;
    double y0 = 2*(y-cy) / height;
    return (x0*x0 + y0*y0 < 1.0);
  }
  
  /////////////////////
  // private methods //
  /////////////////////
  
  private void setDefaults() {
    this.isFilled = false;
    this.color = Color.black;
    this.fillColor = Color.black;
  }
}
