package breadboards;

/**
 * Class whose objects are rectangle whose width, height, and location coordinates are all stored as doubles
 * @author paul oser
 */
public class GRectangle {
  
  public final double EMPTY_THRESHOLD = 0.00000001;
  
  double x;
  double y;
  double width;
  double height;
  
  /**
   * constructs an empty rectangle (dimensions and coordinates are zero)
   */
  public GRectangle() {}
  
  public GRectangle(double width, double height) {
    this.width = width;
    this.height = height;
  }
  
  /**
   * constructs a rectangle with the specified dimensions at the specified location
   * @param x x-coordinate of specified location
   * @param y y-coordinate of specified location
   * @param width specified width
   * @param height specified height
   */
  public GRectangle(double x, double y, double width, double height) {
    this.x = x;
    this.y = y;
    this.width = width;
    this.height = height;
  }
  
  /**
   * returns true when point specified (i.e., (x,y)) is on or inside the GRectangle 
   * @param x x-coordinate of the point specified
   * @param y y-coordinate of the point specified
   * @return true when (x,y) is on or inside the GRectangle, false otherwise
   */
  public boolean contains(double x, double y) {
    return ((x >= (int) this.getX()) && (x <= (int) this.getX() + (int) this.width) &&
            (y >= (int) this.getY()) && (y <= (int) this.getY() + (int) this.height)); 
  }
  
  /**
   * returns true when point specified is on or inside the GRectangle
   * @param p the point specified
   * @return true when point specified is on or inside the GRectangle, false otherwise
   */
  public boolean contains(GPoint p) {
    return contains(p.getX(),p.getY());
  }
  
  public boolean intersects(GRectangle otherRectangle) {
    double xMin = otherRectangle.getX();
    double xMax = xMin + otherRectangle.getWidth();
    double yMin = otherRectangle.getY();
    double yMax = yMin + otherRectangle.getHeight();
    return (this.contains(xMin,yMin) ||
            this.contains(xMin,yMax) ||
            this.contains(xMax,yMin) ||
            this.contains(xMax,yMax));
  }
  
  /**
   * returns the width of the GRectangle
   * @return the width of the GRectangle
   */
  public double getWidth() {
    return this.width;
  }
  
  /**
   * returns the height of the GRectangle
   * @return the height of the GRectangle
   */
  public double getHeight() {
    return this.height;
  }
  
  /**
   * returns the location of this GRectangle
   * @return the location of this GRectangle
   */
  public GPoint getLocation() {
    return new GPoint(this.x,this.y);
  }
  
  /**
   * returns the dimensions of this GRectangle
   * @return the dimensions of this GRectangle
   */
  public GDimension getSize() {
    return new GDimension(this.width,this.height);
  }
  
  /** 
   * returns the x-coordinate of the upper left corner of this rectangle
   * @return the x-coordinate of the upper left corner of this rectangle
   */
  public double getX() {
    return this.x;
  }
  
  /** 
   * returns the y-coordinate of the upper left corner of this rectangle
   * @return the y-coordinate of the upper left corner of this rectangle
   */
  public double getY() {
    return this.y;
  }
  
  /**
   * returns whether or not the width and height are essentially zero (i.e., sufficiently small)
   * @return true if both width and height are less than GRectangle.EMPTY_THRESHOLD, false otherwise
   */
  public boolean isEmpty() {
    return ((Math.abs(width) < EMPTY_THRESHOLD) && 
            (Math.abs(height) < EMPTY_THRESHOLD));
  }
  
  /**
   * sets the location of the upper left corner of the rectangle to the specified coordinates
   * @param x x-coordinate specified
   * @param y y-coordinate specified
   */
  public void setLocation(double x, double y) {
    this.x = x;
    this.y = y;
  }
  
  /**
   * sets the dimensions of the rectangle to those specified
   * @param width the specified width
   * @param height the specified height
   */
  public void setSize(double width, double height) {
    this.width = width;
    this.height = height;
  }
  
  /** 
   * sets the dimensions of the rectangle to the specified dimensions
   * @param size the specified dimensions
   */
  public void setSize(GDimension size) {
    this.width = size.getWidth();
    this.height = size.getHeight();
  }
  
  /** 
   * grows the dimensions of the rectangle (width by dx, height by dy)
   * @param dx amount to add to the width of the rectangle
   * @param dy amount to add to the height of the rectangle
   */
  public void grow(double dx, double dy) {
    this.setSize(this.getWidth() + dx, this.getHeight() + dy);
  }
  
  /**
   * translates the rectangle, adding dx to the x-coordinate and dy to the y-coordinate of its upper left corner
   * @param dx amount to add to the x-coordinate of its upper left corner
   * @param dy amount to add to the y-coordinate of its upper left corner
   */
  public void translate(double dx, double dy) {
    this.setLocation(this.getX() + dx, this.getY() + dy);
  }
  
  /**
   * returns the GRectangle that bounds both this GRectangle and another specified GRectangle
   * @param r2 the GRectangle specified
   * @return the GRectangle that bounds both this GRectangle and r2
   */
  public GRectangle union(GRectangle r2) {
    double minX = Math.min(this.getX(), r2.getX());
    double maxX = Math.max(this.getX() + this.getWidth(), r2.getX() + r2.getWidth());
    double minY = Math.min(this.getY(), r2.getY());
    double maxY = Math.max(this.getY() + this.getHeight(), r2.getY() + r2.getHeight());
    return (new GRectangle(minX,minY,maxX-minX,maxY-minY));
  }
  
  public String toString() {
    return "GRectangle[(x,y)="+this.getX() + "," + this.getY() + 
           " dim = " + this.getWidth() + "x" + this.getHeight();        
  }

}
