package breadboards;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.util.ArrayList;

import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.Timer;

/**
 * A JFrame with several components pre-added for quick development 
 * (i.e., a text area, text field, label, two buttons, 
 * and a panel on which drawings can be made. 
 * This class includes methods for taking advantage of a built in timer,
 * to facilitate animations.
 * @author paul oser
 */
public class Breadboard extends JFrame 
                        implements GObjectParent {
  
  private static final long serialVersionUID = -5450625255869548740L;
  ///////////////
  // constants //
  ///////////////
  
  /**
   * An identifier for the north (top) panel, where the text area resides
   */
  public final static int NORTH = 0;
  
  /**
   * An identifier for the center panel, intended for drawing GObjects
   */
  public final static int CENTER = 1;
  
  /** An identifier for the south (bottom) panel, where the label, text
   * field, and both buttons reside.
   */
  public final static int SOUTH = 2;
  
  ////////////////////////
  // instance variables //
  ////////////////////////
  
  int                breadboardType;
  
  JPanel             northPanel;
  BCanvas            centerPanel;
  JPanel             southPanel;
  
  JTextArea          textArea;
  JLabel             label;
  JTextField         textField;
  JButton            button1;
  JButton            button2;
  Timer              timer;
  Timer              repaintTimer;
  
  Color              backgroundColor = Color.BLACK;
  
  ArrayList<GObject> gObjects;
  
  private GObject    gObjectUnderMouse;
  
  ///////////////////
  // inner classes //
  ///////////////////
  
  class BCanvas extends JPanel {

    private static final long serialVersionUID = -5916853238873818712L;

    public void paintComponent(Graphics g) {
      super.paintComponent(g);
      
      Color prevColor = g.getColor();
      g.setColor(backgroundColor);
      g.fillRect(0, 0, this.getWidth(), this.getHeight()); // fill a rectangle with background color
      g.setColor(prevColor);
      
      for (GObject gObject : gObjects) {
        if (gObject.isVisible())
          gObject.draw(g);
      }
    }
  }
  
  //////////////////
  // constructors //
  //////////////////
  
  /**
   * This constructs a 500 x 500 breadboard window
   */
  public Breadboard() {
    super("My Program");
    this.setSize(500,500);
    addComponents();
    addMouseAndMouseMotionListeners();
    addTimers();
    this.addWindowListener(new WindowListener() {

      @Override
      public void windowOpened(WindowEvent e) {
        Breadboard.this.repaintTimer.start();
      }

      public void windowClosing(WindowEvent e) {}
      public void windowClosed(WindowEvent e) {}
      public void windowIconified(WindowEvent e) {}
      public void windowDeiconified(WindowEvent e) {}
      public void windowActivated(WindowEvent e) {}
      public void windowDeactivated(WindowEvent e) {}});
  }
  
  /** 
   * This constructs a breadboard window with the specified dimensions
   * @param width the width of the window produced
   * @param height the height of the window produced
   */
  public Breadboard(int width, int height) {
    super("My Program");
    this.setSize(width,height);
    addComponents();
    addMouseAndMouseMotionListeners();
    addTimers();
    this.addWindowListener(new WindowListener() {

      @Override
      public void windowOpened(WindowEvent e) {
        Breadboard.this.repaintTimer.start();
      }

      public void windowClosing(WindowEvent e) {}
      public void windowClosed(WindowEvent e) {}
      public void windowIconified(WindowEvent e) {}
      public void windowDeiconified(WindowEvent e) {}
      public void windowActivated(WindowEvent e) {}
      public void windowDeactivated(WindowEvent e) {}});
  }
  
  ////////////////////
  // public methods //
  ////////////////////
  
  public void pause(int milliseconds) {
    try {
      Thread.sleep(milliseconds);  
    } catch (InterruptedException e) {
      e.printStackTrace();
    }
  }
  
  /** This sets the background color of the breadboard's content area (i.e, the drawing area in 
   * the center)
   */
  public void setBackground(Color color) {
    backgroundColor = color;
  }
  
  /**
   * returns a reference to the breadboard
   * @return a reference to the breadboard
   */
  public Breadboard getBreadboard() {
    return this;
  }
  
  /** 
   * returns the text area at the top of the breadboard window
   * @return the text area at the top of the breadboard window
   */
  public JTextArea getTextArea() {
    return textArea;
  }
  
  /**
   * returns a reference to the label in the south panel of this breadboard
   * @return a reference to the label in the south panel of this breadboard
   */
  public JLabel getLabel() {
    return label;
  }
  
  /**
   * returns a reference to the text field in the south panel of this breadboard
   * @return a reference to the text field in the south panel of this breadboard
   */
  public JTextField getTextField() {
    return textField;
  }
  
  /**
   * returns a reference to the built-in timer associated with this breadboard
   * @return a reference to the built-in timer associated with this breadboard
   */
  public Timer getTimer() {
    return timer;
  }
  
  /**
   * returns a reference to the left button in the south panel of this breadboard
   * @return a reference to the left button in the south panel of this breadboard
   */
  public JButton getButton1() {
    return button1;
  }
  
  /**
   * returns a reference to the right button in the south panel of this breadboard
   * @return a reference to the right button in the south panel of this breadboard
   */
  public JButton getButton2() {
    return button2;
  }
  
  /**
   * returns a reference to one of the panels of this breadboard (as indicated by the panelConstant)
   * @param panelConstant (e.g., Breadboard.NORTH, Breadboard.CENTER, Breadboard.SOUTH)
   * @return a reference to one of the panels of this breadboard (as indicated by the panelConstant)
   */
  public JPanel getPanel(int panelConstant) {
    switch (panelConstant) {
    case Breadboard.NORTH : return this.northPanel;
    case Breadboard.CENTER : return this.centerPanel;
    case Breadboard.SOUTH : return this.southPanel;
    }
    return null;  
  }
  
  /**
   * returns the height of the drawable area in the center of this breadboard
   * @return the height of the drawable area in the center of this breadboard
   */
  public int getCenterPanelHeight() {
    int northPanelHeight = this.getPanel(Breadboard.NORTH).getHeight();
    int southPanelHeight = this.getPanel(Breadboard.SOUTH).getHeight();
    int otherJFrameElementsHeight = this.getHeight() - this.getContentPane().getHeight();
    return this.getHeight() - 
           northPanelHeight -
           southPanelHeight -
           otherJFrameElementsHeight;
  }
  
  /**
   * returns the width of the drawable area in the center of this breadboard
   * @return the width of the drawable area in the center of this breadboard
   */
  public int getCenterPanelWidth() {
    int otherJFrameElementsWidth = this.getWidth() - this.getContentPane().getWidth();
    return this.getWidth() - otherJFrameElementsWidth;
  }
  
  /**
   * adds the specified GObject to this breadboard
   * @param gObject the GObject to be added
   */
  public void add(GObject gObject) {
    this.gObjects.add(gObject);
    gObject.setParent(this);
    this.repaint();
  }
  
  /**
   * adds the specified GObject to this breadboard at the specified location 
   * @param gObject the specified GObject
   * @param x the x-coordinate of the location specified
   * @param y the y-coordinate of the location specified
   */
  public void add(GObject gObject, double x, double y) {
    gObject.setLocation(x, y);
    gObject.setParent(this);
    this.gObjects.add(gObject);
  }
  
  /** 
   * adds the specified GObject to this breadboard at the specified point 
   * @param gObject the specified GObject
   * @param pt the specified point
   */
  public void add(GObject gObject, GPoint pt) {
    gObject.setLocation(pt.getX(),pt.getY());
    gObject.setParent(this);
    this.gObjects.add(gObject);
  }
  
  /**
   * removes the specified GObject from the breadboard
   * @param gObject the specified GObject
   */
  public void remove(GObject gObject) {
    this.gObjects.remove(gObject);
    this.repaint();
  }
  
  /**
   * removes all GObjects currently on the breadboard 
   */
  public void removeAll() {
    for (int i = this.getElementCount()-1; i >= 0; i--) {
      this.gObjects.remove(i);
    }
    this.repaint();
  }
  
  /**
   * returns the top-most element at the specified location
   * @param x the x-coordinate of the specified location
   * @param y the y-coordinate of the specified location
   * @return a reference to the top-most (in the z-order sense) element at the specified location
   */
  public GObject getElementAt(double x, double y) {
    GObject objectClicked = null;
    for (GObject gObject : gObjects) {
      if (gObject.contains(x,y)) {
        objectClicked = gObject;
      }
    }
    return objectClicked;
  }
  
  /** 
   * returns the top-most element at the specified point
   * @param pt the specified point
   * @return a reference to the top-most (in the z-order sense) element at the specified point
   */
  public GObject getElementAt(GPoint pt) {
    return getElementAt(pt.getX(),pt.getY());
  }
  
  /**
   * returns the number of GObjects on the breadboard
   * @return the number of GObjects on the breadboard
   */
  public int getElementCount() {
    return gObjects.size();
  }
  
  /**
   * returns the element with the specified z-level.
   * @param index the position in the z-order list maintained by the breadboard
   * @return a reference to the GObject with the specified index (i.e., position in the aforementioned list)
   */
  public GObject getElement(int index) {
    return gObjects.get(index);
  }
  
  /**
   * Swaps the z-orders of the specified GObject and the object immediately "above" it.
   * @param gObj the specified GObject
   */
  public void sendForward(GObject gObj) {
    int gObjIndex = gObjects.indexOf(gObj);
    if (gObjIndex < gObjects.size() - 1) {
      GObject nextHigherObject = gObjects.get(gObjIndex+1);
      gObjects.set(gObjIndex, nextHigherObject);
      gObjects.set(gObjIndex+1, gObj);
      repaint();
    }
  }
  
  /**
   * Swaps the z-orders of the specified GObject and the object immediately "below" it.
   * @param gObj the specified GObject
   */
  public void sendBackward(GObject gObj) {
    int gObjIndex = gObjects.indexOf(gObj);
    if (gObjIndex > 0) {
      GObject nextLowerObject = gObjects.get(gObjIndex-1);
      gObjects.set(gObjIndex, nextLowerObject);
      gObjects.set(gObjIndex-1, gObj);
      repaint();
    }
  }
  
  /** repaints the breadboard
   */
  public void updateDisplay() {
    this.getContentPane().setBackground(backgroundColor);
    repaint();
  }
  
  /**
   * Moves the specified object "on top" of all other GObjects on the breadboard
   * @param gObj the specified GObject
   */
  public void sendToFront(GObject gObj) {
    gObjects.remove(gObj);
    gObjects.add(gObj);
    repaint();
  }
  
  /**
   * Moves the specified object below all other GObjects on the breadboard.
   * @param gObj the specified GObject
   */
  public void sendToBack(GObject gObj) {
    gObjects.remove(gObj);
    gObjects.add(0, gObj);
    repaint();
  }
  
  /**
   * Expands or contracts the breadboard window size so that its central drawing area has the specified dimensions
   * @param width the new width of the central drawing area
   * @param height the new height of the central drawing area
   */
  public void setCenterPanelSize(int width, int height) {
    int cw = this.getCenterPanelWidth();
    int ch = this.getCenterPanelHeight();
    int ww = this.getWidth();
    int wh = this.getHeight();
    this.setSize(ww + (width-cw), wh + (height-ch));
  }
 
  
  /////////////////////
  // private methods //
  /////////////////////
  
  private void addComponents() {
    textArea = new JTextArea(4,40);
    textArea.setEnabled(false);

    label = new JLabel("Input:");
    textField = new JTextField(20);
    button1 = new JButton("Btn1");
    button2 = new JButton("Btn2");
    
    gObjects = new ArrayList<GObject>();
    
    this.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    
    northPanel = new JPanel();
    northPanel.add(textArea);
    this.getContentPane().add("North", northPanel);
    
    centerPanel = new BCanvas();
    this.getContentPane().add("Center", centerPanel);
    
    southPanel = new JPanel();
    southPanel.add(label);
    southPanel.add(textField);
    
    // build and add button1...
    button1.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        onButton1Click();
      }     
    });
    southPanel.add(button1);
    
    // build and add button2...
    button2.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        onButton2Click();
      }     
    });
    southPanel.add(button2);
    this.getContentPane().add("South", southPanel);
  }
  
  private void addMouseAndMouseMotionListeners() {
    
    centerPanel.addMouseListener(new MouseListener() {

      @Override
      public void mouseClicked(MouseEvent e) {
        gObjectUnderMouse = getElementAt(e.getX(),e.getY());
        if (gObjectUnderMouse != null) {
          for (MouseListener ml : gObjectUnderMouse.getMouseListeners()) {
            ml.mouseClicked(e);
          }
        } 
      }

      @Override
      public void mousePressed(MouseEvent e) {
        gObjectUnderMouse = getElementAt(e.getX(),e.getY());
        if (gObjectUnderMouse != null) {
          for (MouseListener ml : gObjectUnderMouse.getMouseListeners()) {
            ml.mousePressed(e);
          }
        } 
      }

      @Override
      public void mouseReleased(MouseEvent e) {
        gObjectUnderMouse = getElementAt(e.getX(),e.getY());
        if (gObjectUnderMouse != null) {
          for (MouseListener ml : gObjectUnderMouse.getMouseListeners()) {
            ml.mouseReleased(e);
          }
        } 
      }

      @Override
      public void mouseEntered(MouseEvent e) {
        // TODO: not yet implemented  
      }

      @Override
      public void mouseExited(MouseEvent e) {
        // TODO: not yet implemented 
      }});
    
    // deal with mouse motion listeners
    centerPanel.addMouseMotionListener(new MouseMotionListener() {

      @Override
      public void mouseDragged(MouseEvent e) {
        GObject gObjectInitiallyUnderMouse = gObjectUnderMouse;
        gObjectUnderMouse = getElementAt(e.getX(),e.getY());
        if (gObjectUnderMouse != gObjectInitiallyUnderMouse) {
          if (gObjectUnderMouse != null) {
            for (MouseListener ml : gObjectUnderMouse.getMouseListeners()) {
              ml.mouseEntered(e);
            }
          }
          if (gObjectInitiallyUnderMouse != null) {
            for (MouseListener ml : gObjectInitiallyUnderMouse.getMouseListeners()) {
              ml.mouseExited(e);
            }
          }
        }
        else { // when.. gObjectUnderMouse == gObjectInitiallyUnderMouse
          if (gObjectUnderMouse != null) {
            for (MouseMotionListener mml : gObjectUnderMouse.getMouseMotionListeners()) {
              mml.mouseDragged(e);
            }
          }
        }
      }

      @Override
      public void mouseMoved(MouseEvent e) {
        GObject gObjectInitiallyUnderMouse = gObjectUnderMouse;
        gObjectUnderMouse = getElementAt(e.getX(),e.getY());
        if (gObjectUnderMouse != gObjectInitiallyUnderMouse) {
          if (gObjectUnderMouse != null) {
            for (MouseListener ml : gObjectUnderMouse.getMouseListeners()) {
              ml.mouseEntered(e);
            }
          }
          if (gObjectInitiallyUnderMouse != null) {
            for (MouseListener ml : gObjectInitiallyUnderMouse.getMouseListeners()) {
              ml.mouseExited(e);
            }
          }
        }
        else { // when.. gObjectUnderMouse == gObjectInitiallyUnderMouse
          if (gObjectUnderMouse != null) {
            for (MouseMotionListener mml : gObjectUnderMouse.getMouseMotionListeners()) {
              mml.mouseMoved(e);
            }
          }
        }
      }});
    
    this.init();
    this.setVisible(true);
  }
  
  private void addTimers() {
    this.timer = new Timer(100,new ActionListener() {

      @Override
      public void actionPerformed(ActionEvent e) {
        Breadboard.this.onTimerTick();
      }
      
    });
    
    
    this.repaintTimer = new Timer(1,new ActionListener() {

      @Override
      public void actionPerformed(ActionEvent e) {
        updateDisplay();
      }   
    });
    
  }
  
  ////////////////////////////////////////
  // methods to be overloaded by client //
  ////////////////////////////////////////
  
  /** 
   * Override to perform some action before the breadboard is initially made visible
   */
  public void init() {};
  
  /**
   * Override to perform some action whenever the left button is clicked
   */
  public void onButton1Click() {}; 
  
  /** 
   * Override to perform some action whenever the right button is clicked.
   */
  public void onButton2Click() {};
  
  /**
   * Override to perform some action with each tick of the timer, provided the timer has been started.
   */
  public void onTimerTick() {};

}
