package breadboards;

import java.awt.Color;
import java.awt.Graphics;

/**
 * Subclass of GObject suitable for displaying rectangles
 * @author paul oser
 */
public class GRect extends GObject {
  
  double width;
  double height;
  
  boolean isFilled;
  Color color;
  Color fillColor;

  /**
   * constructs a GRect with specified width and height with upper left corner at (0,0)
   * @param width the specified width
   * @param height the specified height
   */
  public GRect(double width, double height) {
    this.width = width;
    this.height = height;
    setDefaults();
  }
  
  /**
   * constructs a GRect with a specified width and height at a specified location
   * @param x the x-coordinate of the upper left corner of the rectangle
   * @param y the y-coordinate of the upper left corner of the rectangle
   * @param width the specified width
   * @param height the specified height
   */
  public GRect(double x, double y, double width, double height) {
    this.x = x;
    this.y = y;
    this.width = width;
    this.height = height;
    setDefaults();
  }
  
  /////////////
  // Getters //
  /////////////
  
  /**
   * returns the bounding rectangle for this GRect
   * @return the bounding rectangle for this GRect
   */
  public GRectangle getBounds() {
    return new GRectangle(this.x, this.y, this.width, this.height);
  }
  
  /**
   * returns the border color of this GRect
   * @return the border color of this GRect
   */
  public Color getColor() {
    return this.color;
  }
  
  /**
   * returns the fill color for this GRect
   * @return the fill color for this GRect
   */
  public Color getFillColor() {
    return this.color;
  }
  
  /**
   * returns the height of this GRect
   * @return the height of this GRect
   */
  public double getHeight() {
    return this.height;
  }
  
  /**
   * returns the width of this GRect
   * @return the width of this GRect
   */
  public double getWidth() {
    return this.width;
  }
  
  /**
   * returns whether or not this GRect should be displayed with an opaque interior 
   * @return true if an opaque interior should be used, false otherwise
   */
  public boolean isFilled() {
    return this.isFilled;
  }
  
  /////////////
  // Setters //
  /////////////
  
  /**
   * sets the border color of the GRect to the specified color
   * @param color the specified color
   */
  public void setColor(Color color) {
    this.color = color;
  }
  
  /**
   * this must be called for the GRect to show an opaque interior
   * @param isFilled true if an opaque interior is to be displayed, false otherwise
   */
  public void setFilled(boolean isFilled) {
    this.isFilled = isFilled;
  }
  
  /**
   * set the fill color for the opaque interior to the specified color
   * @param fillColor the specified color
   */
  public void setFillColor(Color fillColor) {
    this.fillColor = fillColor;
  }
  
  /** 
   * set the width and height of the GRect to the specified values
   * @param width the specified width
   * @param height the specified height
   */
  public void setSize(double width, double height) {
    this.width = width;
    this.height = height;
  }
  
  /**
   * set the width, height, and location of the GRect to the specified values
   * @param x x-coordinate of the upper left corner of the GRect
   * @param y y-coordinate of the upper left corner of the GRect
   * @param width width the specified width
   * @param height height the specified height
   */
  public void setBounds(double x, double y, double width, double height) {
    this.x = x;
    this.y = y;
    this.width = width;
    this.height = height;
  }
  
  /**
   * set the width, height, and location of the GRect to those associated with the specified bounding rectangle 
   * @param bounds a bounding rectangle for the GRect with the same dimensions and location
   */
  public void setBounds(GRectangle bounds) {
    this.setBounds(bounds.getX(),bounds.getY(),bounds.getWidth(),bounds.getHeight());
  }
  
  ///////////////////
  // other methods //
  ///////////////////
  
  /**
   * returns true if (x,y) is inside this GRect
   * @param x the x-coordinate of the point in question
   * @param y the y-coordinate of the point in question
   * @return true if (x,y) is inside this GRect, false otherwise
   */
  public boolean contains(double x, double y) {
    return this.getBounds().contains(x, y);
  }
  
  /**
   * scale the width of the GRect by a factor of sx and the height of the GRect by a factor of sy
   * @param sx the horizontal scaling factor
   * @param sy the vertical scaling factor
   */
  public void scale(double sx, double sy) {
    this.width *= sx;
    this.height *= sy;
  }
  
  /** 
   * scale this GRect by a factor of sf
   * @param sf the scaling factor
   */
  public void scale(double sf) {
    this.scale(sf,sf);
  }
  
  /** 
   * draws the GRect (generally not called directly)
   */
  public void draw(Graphics g) {
    
    Color originalFillColor = g.getColor();
    
    if (this.isFilled) {
      g.setColor(fillColor);
      g.fillRect((int) x, (int) y, (int) width, (int) height); 
    }
    
    g.setColor(color);
    g.drawRect((int) x, (int) y, (int) width, (int) height); 
    
    g.setColor(originalFillColor);

  }
  
  /////////////////////
  // private methods //
  /////////////////////
  
  private void setDefaults() {
    this.isFilled = false;
    this.color = Color.black;
    this.fillColor = Color.black;
  }
  
}
